const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const axios = require('axios');
const cheerio = require('cheerio');
const puppeteer = require('puppeteer');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 5000;

// Security middleware
app.use(helmet());

// CORS configuration for React frontend and Flutter WebView
app.use(cors({
  origin: [
    'http://localhost:3000', 
    'http://192.168.18.13:3000',
    'http://10.0.2.2:3000',  // Flutter WebView (Android emulator)
    'http://127.0.0.1:3000',   // Alternative localhost
    'http://localhost:5000',
    'http://192.168.18.13:5000',
    'http://10.0.2.2:5000'
  ],
  credentials: true,
  methods: ['GET', 'POST', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With']
}));

// Serve static files
app.use(express.static('public'));

// Debugging middleware
app.use((req, res, next) => {
  console.log(`${new Date().toISOString()} - ${req.method} ${req.url} from ${req.get('Origin') || 'unknown origin'}`);
  next();
});

// Body parsing middleware
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Instagram video download endpoint
app.post('/api/download-instagram', async (req, res) => {
  try {
    const { url, type = 'post' } = req.body;

    if (!url) {
      return res.status(400).json({
        success: false,
        error: 'Instagram URL is required'
      });
    }

    if (!isValidInstagramUrl(url)) {
      return res.status(400).json({
        success: false,
        error: 'Invalid Instagram URL format'
      });
    }

    console.log(`Processing Instagram URL: ${url} (Type: ${type})`);

    // Try different methods to extract video info
    let videoInfo = null;
    let methodUsed = '';

    // Method 1: Try with enhanced Puppeteer
    try {
      console.log('Trying Method 1: Enhanced Puppeteer...');
      videoInfo = await extractWithEnhancedPuppeteer(url, type);
      if (videoInfo && videoInfo.length > 0) {
        methodUsed = 'Enhanced Puppeteer';
        console.log('Enhanced Puppeteer method successful');
      }
    } catch (error) {
      console.log('Enhanced Puppeteer method failed:', error.message);
    }

    // Method 2: Try with Instagram embed API
    if (!videoInfo || videoInfo.length === 0) {
      try {
        console.log('Trying Method 2: Instagram embed API...');
        videoInfo = await extractWithInstagramEmbed(url, type);
        if (videoInfo && videoInfo.length > 0) {
          methodUsed = 'Instagram Embed API';
          console.log('Instagram embed method successful');
        }
      } catch (error) {
        console.log('Instagram embed method failed:', error.message);
      }
    }

    // Method 3: Try with Cheerio (fallback)
    if (!videoInfo || videoInfo.length === 0) {
      try {
        console.log('Trying Method 3: Cheerio fallback...');
        videoInfo = await extractWithCheerio(url, type);
        if (videoInfo && videoInfo.length > 0) {
          methodUsed = 'Cheerio Fallback';
          console.log('Cheerio method successful');
        }
      } catch (error) {
        console.log('Cheerio method failed:', error.message);
      }
    }

    // Method 4: Try with direct Instagram API approach
    if (!videoInfo || videoInfo.length === 0) {
      try {
        console.log('Trying Method 4: Direct Instagram API...');
        videoInfo = await extractWithDirectAPI(url, type);
        if (videoInfo && videoInfo.length > 0) {
          methodUsed = 'Direct Instagram API';
          console.log('Direct Instagram API method successful');
        }
      } catch (error) {
        console.log('Direct Instagram API method failed:', error.message);
      }
    }

    if (!videoInfo || videoInfo.length === 0) {
      console.log('All extraction methods failed');
      return res.status(404).json({
        success: false,
        error: 'Could not extract video information. The content might be private, not available, or Instagram has updated their security measures.',
        details: {
          methodsTried: ['Enhanced Puppeteer', 'Instagram Embed API', 'Cheerio Fallback', 'Direct Instagram API'],
          suggestions: [
            'Try with a different Instagram post (some posts may be private)',
            'Ensure the post is public and accessible',
            'Try refreshing the Instagram page first',
            'Check if the post contains video content'
          ]
        }
      });
    }

    // Add download instructions and warnings
    const enhancedVideoInfo = videoInfo.map(video => ({
      ...video,
      downloadNote: 'Note: Instagram CDN URLs may have access restrictions. Try downloading in a new tab or use the copy URL feature.',
      alternativeMethods: [
        'Right-click the video and "Save as..."',
        'Use browser developer tools to download',
        'Try opening the URL in a new incognito tab',
        'Use the "Play in New Tab" feature for better compatibility'
      ]
    }));

    res.json({
      success: true,
      data: enhancedVideoInfo,
      methodUsed: methodUsed,
      warnings: [
        'Instagram CDN URLs are protected and may require proper referer headers',
        'Some videos may not download directly due to Instagram\'s security measures',
        'Try using the "Play in New Tab" feature for better compatibility',
        'If download fails, try the alternative methods provided'
      ]
    });

  } catch (error) {
    console.error('Instagram download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to process Instagram URL. Please try again.',
      details: error.message
    });
  }
});

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({
    success: true,
    message: 'Instagram Downloader Backend is running',
    timestamp: new Date().toISOString()
  });
});



// Instagram URL validation
function isValidInstagramUrl(url) {
  const instagramRegex = /^https?:\/\/(www\.)?instagram\.com\/(p|reel|tv|stories)\/[a-zA-Z0-9_-]+\/?/;
  return instagramRegex.test(url);
}

// Method 1: Enhanced Puppeteer extraction
async function extractWithEnhancedPuppeteer(url, type) {
  let browser;
  try {
    browser = await puppeteer.launch({
      headless: "new",
      args: [
        '--no-sandbox',
        '--disable-setuid-sandbox',
        '--disable-dev-shm-usage',
        '--disable-accelerated-2d-canvas',
        '--no-first-run',
        '--no-zygote',
        '--disable-gpu',
        '--disable-web-security',
        '--disable-features=VizDisplayCompositor',
        '--user-agent=Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
      ]
    });

    const page = await browser.newPage();
    
    // Set viewport and user agent
    await page.setViewport({ width: 1280, height: 720 });
    
    // Set extra headers
    await page.setExtraHTTPHeaders({
      'Accept-Language': 'en-US,en;q=0.9',
      'Accept-Encoding': 'gzip, deflate, br',
      'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8',
      'Cache-Control': 'no-cache',
      'Pragma': 'no-cache',
      'Referer': 'https://www.instagram.com/'
    });

    // Navigate to the Instagram page
    console.log('Navigating to Instagram page...');
    await page.goto(url, { 
      waitUntil: 'networkidle2',
      timeout: 30000 
    });

    // Wait for content to load
    console.log('Waiting for content to load...');
    await page.waitForTimeout(5000);

    // Try to find video elements
    const videoInfo = await page.evaluate((contentType) => {
      const videos = [];
      const images = [];
      
      console.log('Searching for video elements...');
      
      // Look for video elements with src
      const videoElements = document.querySelectorAll('video');
      console.log(`Found ${videoElements.length} video elements`);
      
      videoElements.forEach((video, index) => {
        if (video.src) {
          videos.push({
            quality: `HD (${index + 1})`,
            size: 'Unknown',
            format: 'MP4',
            url: video.src,
            thumbnail: video.poster || '',
            type: 'video'
          });
        }
      });

      // Look for source elements inside video tags
      const sourceElements = document.querySelectorAll('video source');
      console.log(`Found ${sourceElements.length} source elements`);
      
      sourceElements.forEach((source, index) => {
        if (source.src) {
          videos.push({
            quality: `Quality ${index + 1}`,
            size: 'Unknown',
            format: 'MP4',
            url: source.src,
            thumbnail: '',
            type: 'video'
          });
        }
      });

      // Look for image elements (for posts without videos)
      const imageElements = document.querySelectorAll('img[src*="instagram"]');
      console.log(`Found ${imageElements.length} image elements`);
      
      imageElements.forEach((img, index) => {
        if (img.src && img.src.includes('instagram')) {
          images.push({
            quality: `Image (${index + 1})`,
            size: 'Unknown',
            format: 'JPEG',
            url: img.src,
            thumbnail: img.src,
            type: 'image'
          });
        }
      });

      // Return videos if available, otherwise images
      return videos.length > 0 ? videos : images;
    }, type);

    await browser.close();
    return videoInfo;

  } catch (error) {
    console.error('Enhanced Puppeteer extraction error:', error);
    if (browser) {
      await browser.close();
    }
    throw error;
  }
}

// Method 2: Instagram embed API
async function extractWithInstagramEmbed(url, type) {
  try {
    // Convert to embed URL
    const embedUrl = url.replace('/reel/', '/p/').replace('/tv/', '/p/') + '/embed/';
    console.log('Trying embed URL:', embedUrl);
    
    const response = await axios.get(embedUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
        'Accept-Language': 'en-US,en;q=0.9',
        'Accept-Encoding': 'gzip, deflate, br',
        'Referer': 'https://www.instagram.com/',
        'Sec-Fetch-Dest': 'document',
        'Sec-Fetch-Mode': 'navigate',
        'Sec-Fetch-Site': 'same-origin'
      },
      timeout: 15000
    });

    const $ = cheerio.load(response.data);
    const videos = [];

    // Look for video sources in the embed
    $('video source').each((index, element) => {
      const src = $(element).attr('src');
      if (src) {
        videos.push({
          quality: `Quality ${index + 1}`,
          size: 'Unknown',
          format: 'MP4',
          url: src,
          thumbnail: '',
          type: 'video'
        });
      }
    });

    // Look for video URLs in script tags
    $('script').each((index, element) => {
      const content = $(element).html();
      if (content) {
        // Look for various video URL patterns
        const patterns = [
          /"video_url":"([^"]+)"/,
          /"video_url":"([^"]+)"/,
          /video_url=([^&]+)/,
          /"contentUrl":"([^"]+)"/,
          /"url":"([^"]*\.mp4[^"]*)"/
        ];

        patterns.forEach(pattern => {
          const match = content.match(pattern);
          if (match && match[1]) {
            const videoUrl = match[1].replace(/\\u0026/g, '&').replace(/\\/g, '');
            if (videoUrl && !videos.some(v => v.url === videoUrl)) {
              videos.push({
                quality: 'HD',
                size: 'Unknown',
                format: 'MP4',
                url: videoUrl,
                thumbnail: '',
                type: 'video'
              });
            }
          }
        });
      }
    });

    return videos.length > 0 ? videos : null;

  } catch (error) {
    console.error('Instagram embed extraction error:', error);
    throw error;
  }
}

// Method 3: Extract with Cheerio (fallback method)
async function extractWithCheerio(url, type) {
  try {
    console.log('Trying direct URL with Cheerio...');
    
    const response = await axios.get(url, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
        'Accept-Language': 'en-US,en;q=0.9',
        'Accept-Encoding': 'gzip, deflate, br',
        'Connection': 'keep-alive',
        'Upgrade-Insecure-Requests': '1',
        'Sec-Fetch-Dest': 'document',
        'Sec-Fetch-Mode': 'navigate',
        'Sec-Fetch-Site': 'none',
        'Referer': 'https://www.instagram.com/'
      },
      timeout: 15000
    });

    const $ = cheerio.load(response.data);
    const videos = [];

    // Look for video sources in the HTML
    $('video source').each((index, element) => {
      const src = $(element).attr('src');
      if (src) {
        videos.push({
          quality: `Quality ${index + 1}`,
          size: 'Unknown',
          format: 'MP4',
          url: src,
          thumbnail: '',
          type: 'video'
        });
      }
    });

    // Look for video URLs in script tags
    $('script').each((index, element) => {
      const content = $(element).html();
      if (content && content.includes('video_url')) {
        try {
          const patterns = [
            /"video_url":"([^"]+)"/,
            /video_url=([^&]+)/,
            /"contentUrl":"([^"]+)"/,
            /"url":"([^"]*\.mp4[^"]*)"/
          ];

          patterns.forEach(pattern => {
            const match = content.match(pattern);
            if (match && match[1]) {
              const videoUrl = match[1].replace(/\\u0026/g, '&').replace(/\\/g, '');
              if (videoUrl && !videos.some(v => v.url === videoUrl)) {
                videos.push({
                  quality: 'HD',
                  size: 'Unknown',
                  format: 'MP4',
                  url: videoUrl,
                  thumbnail: '',
                  type: 'video'
                });
              }
            }
          });
        } catch (e) {
          console.log('Error parsing script content');
        }
      }
    });

    return videos.length > 0 ? videos : null;

  } catch (error) {
    console.error('Cheerio extraction error:', error);
    throw error;
  }
}

// Method 4: Direct Instagram API approach
async function extractWithDirectAPI(url, type) {
  try {
    console.log('Trying direct Instagram API approach...');
    
    // Try to access the post data directly
    const postId = url.match(/\/(p|reel|tv)\/([a-zA-Z0-9_-]+)/)?.[2];
    if (!postId) {
      throw new Error('Could not extract post ID from URL');
    }

    // Try to get the post data
    const apiUrl = `https://www.instagram.com/api/v1/media/${postId}/info/`;
    
    const response = await axios.get(apiUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'Accept': 'application/json',
        'Accept-Language': 'en-US,en;q=0.9',
        'Referer': 'https://www.instagram.com/',
        'X-Requested-With': 'XMLHttpRequest'
      },
      timeout: 15000
    });

    if (response.data && response.data.items && response.data.items[0]) {
      const item = response.data.items[0];
      const videos = [];

      if (item.video_versions && item.video_versions.length > 0) {
        item.video_versions.forEach((video, index) => {
          videos.push({
            quality: `Quality ${index + 1}`,
            size: 'Unknown',
            format: 'MP4',
            url: video.url,
            thumbnail: item.image_versions2?.candidates?.[0]?.url || '',
            type: 'video'
          });
        });
      }

      return videos.length > 0 ? videos : null;
    }

    return null;

  } catch (error) {
    console.error('Direct Instagram API extraction error:', error);
    throw error;
  }
}

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Unhandled error:', error);
  res.status(500).json({
    success: false,
    error: 'Internal server error'
  });
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({
    success: false,
    error: 'Endpoint not found'
  });
});

// Start server
const server = app.listen(PORT, '0.0.0.0', () => {
  console.log(`🚀 Instagram Downloader Backend running on port ${PORT}`);
  console.log(`📱 Health check: http://localhost:${PORT}/api/health`);
  console.log(`🌐 Network access: http://192.168.18.13:${PORT}/api/health`);
  console.log(`⬇️ Download endpoint: http://localhost:${PORT}/api/download-instagram`);
  
  // Log environment
  console.log(`🌍 Environment: ${process.env.NODE_ENV || 'development'}`);
  console.log(`🔧 Node.js version: ${process.version}`);
});

// Handle server errors
server.on('error', (error) => {
  if (error.syscall !== 'listen') {
    throw error;
  }
  
  switch (error.code) {
    case 'EACCES':
      console.error(`Port ${PORT} requires elevated privileges`);
      process.exit(1);
      break;
    case 'EADDRINUSE':
      console.error(`Port ${PORT} is already in use`);
      process.exit(1);
      break;
    default:
      throw error;
  }
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM received, shutting down gracefully');
  process.exit(0);
});

process.on('SIGINT', () => {
  console.log('SIGINT received, shutting down gracefully');
  process.exit(0);
});
